#include "MinSpan.h"

module MinSpanM {
    provides {
        interface StdControl;
    }
    uses {
        interface Timer;
        interface Leds;
        interface SendMsg as SendHopMsg;
        interface ReceiveMsg as ReceiveHopMsg;
        interface SendMsg as SendAlarmMsg;
        interface ReceiveMsg as ReceiveAlarmMsg;
        interface ADC;
        interface CC2420Control;
        interface Random;
    }
}
implementation {
    uint8_t hopCount, parentAddress;
    uint16_t tsrSampleCounter, displayCounter, blinkCounter;
    uint16_t alarmStatusCounter, alarmResetCounter, rebroadcastCounter, timeoutCounter;
    uint16_t tsr;
    uint8_t backoffSlot;    
    bool localAlarmFlag, hopMsgLocked, alarmMsgLocked, broadcastHopFlag;
    TOS_Msg hmsg, amsg;
    
    // Initialize Components
    command result_t StdControl.init() {
        call Leds.init();
        call Random.init();
        tsr = TSR_INF;
        return SUCCESS;
    }

    // Start things up
    command result_t StdControl.start() {
        tsrSampleCounter = 0;        
        displayCounter = 0;                
        blinkCounter = 0;
        alarmStatusCounter = 0;
        alarmResetCounter = 0;        
        rebroadcastCounter = 0;  
        timeoutCounter = 0;
        backoffSlot = 0;
        
        localAlarmFlag = FALSE;
        hopMsgLocked = FALSE; 
        alarmMsgLocked = FALSE; 
        broadcastHopFlag = FALSE;
        
        if (TOS_LOCAL_ADDRESS == ROOT_ADDRESS) {
            hopCount = 0;
            parentAddress = 0;
        }
        else {
            hopCount = HOP_INF;
        }
        
        call Timer.start(TIMER_REPEAT, 50);
        call CC2420Control.SetRFPower(RESOURCE_NONE, 1);
        return SUCCESS;
    }

    // Stop running
    command result_t StdControl.stop() {
        call Timer.stop();
    }
    
    // Broadcast a new hop structure
    task void broadcastHopCount() {
        HopMsg_t* msg;
        broadcastHopFlag = FALSE;
        if (!hopMsgLocked) {
            hopMsgLocked = TRUE;
            atomic {
                msg = (HopMsg_t*)hmsg.data;
                msg->hopCount = hopCount;
                msg->parentAddress = TOS_LOCAL_ADDRESS;
            }
            call SendHopMsg.send(TOS_BCAST_ADDR, sizeof(HopMsg_t), &hmsg);
        }
    }
    
    // Send an alarm to the parent node
    task void sendAlarm() {
        AlarmMsg_t* alarmMsg;
        if (hopCount != HOP_INF && TOS_LOCAL_ADDRESS != ROOT_ADDRESS && !alarmMsgLocked) {
            atomic {
                alarmMsgLocked = TRUE;
                alarmMsg = (AlarmMsg_t*)amsg.data;
                alarmMsg->parentAddress = parentAddress; 
            }
            call SendAlarmMsg.send(TOS_BCAST_ADDR, sizeof(AlarmMsg_t), &amsg);
        }
    }
    
    // Show hops by blinking green LED (used by children only)
    void showHops() {
        // when you dont know hops, show yellow LED
        if (hopCount == HOP_INF) {
            call Leds.greenOff();
            call Leds.yellowOn();
            return;
        }
        call Leds.yellowOff();
        if (blinkCounter > 2*hopCount + 5) {
            blinkCounter = 0;
            return;
        }
        if (blinkCounter >= 2*hopCount)
            call Leds.greenOff();
        else if (blinkCounter % 2 == 0)
            call Leds.greenOn();
        else
            call Leds.greenOff();    
        blinkCounter++;
    }
    
    // setup a broadcast for hop count
    void setupHopBroadcast() {
        if (!broadcastHopFlag) {
            atomic {
                backoffSlot = call Random.rand(); 
                backoffSlot = backoffSlot%10;
                broadcastHopFlag = TRUE;
            }
        }
    }

    // Determine if anything needs to happen at this time
    event result_t Timer.fired() {
        tsrSampleCounter++;
        displayCounter++; 
        alarmStatusCounter++;
        alarmResetCounter++;
        rebroadcastCounter++; 
        timeoutCounter++;
        
        // is it time to sample TSR sensor?
        // (1000 ms)
        if (tsrSampleCounter == 20) {
            tsrSampleCounter = 0;
            call ADC.getData();
        }
        
        // time to display hops? 
        // (200 ms)
        if (displayCounter == 4) {
            displayCounter = 0;
            showHops();
        }
        
        // is the TSR below threshold?
        // (1300 ms)
        if (alarmStatusCounter == 26) {
            alarmStatusCounter = 0;
            if (tsr < TSR_THRESHOLD) {
                localAlarmFlag = TRUE;
                alarmResetCounter = 0;
                call Leds.redOn();
                post sendAlarm();
            }
            else
                localAlarmFlag = FALSE;
        }
        
        // is it time to reset the alarm status?
        // (2000 ms)
        if (alarmResetCounter == 40) {
            alarmResetCounter = 0;
            call Leds.redOff();                
        }
        
        // if we are trying to broadcast, has random delay ended? 
        // (slot = 50 millisec)
        if (broadcastHopFlag) {
            backoffSlot--;
            if (backoffSlot <= 0)
                post broadcastHopCount();
        }
        
        // is it time to rebroadcast our hopCount?
        // (10000 ms)
        if (rebroadcastCounter == 200) {
            rebroadcastCounter = 0;
            setupHopBroadcast();
        }
        
        // has our hop count timed out?
        // (45000 millisec)
        if (TOS_LOCAL_ADDRESS != ROOT_ADDRESS && timeoutCounter == 900) {
            timeoutCounter = 0;
            hopCount = HOP_INF;
        }
        
        return SUCCESS;
    }
    
    // Receive a hop message
    event TOS_MsgPtr ReceiveHopMsg.receive(TOS_MsgPtr msg) {
        HopMsg_t* hopMsg = (HopMsg_t*)msg->data;
        call Leds.yellowToggle();
        if (TOS_LOCAL_ADDRESS != ROOT_ADDRESS) {
            
            // no need to timeout if parent is still intact
            if ((hopMsg->parentAddress == parentAddress) && ((hopMsg->hopCount+1) == hopCount)) {
                timeoutCounter = 0;
            }
                
            // check if we have a better parent
            if ((hopMsg->hopCount+1) < hopCount) {
                atomic {
                    hopCount = hopMsg->hopCount+1;
                    parentAddress = hopMsg->parentAddress;
                    timeoutCounter = 0;
                    rebroadcastCounter = 0;
                }
                setupHopBroadcast();
            }
        }
        return msg;
    }
    
    // Receive an alarm message
    event TOS_MsgPtr ReceiveAlarmMsg.receive(TOS_MsgPtr msg) {
        AlarmMsg_t* alarmMsg = (AlarmMsg_t*)msg->data;
        call Leds.yellowToggle();
        if (alarmMsg->parentAddress == TOS_LOCAL_ADDRESS) {
            // if we are sending alarms, there is no 
            // reason to process children alarm messages
            call Leds.greenOn();
            if (!localAlarmFlag) {
                call Leds.redOn();
                post sendAlarm();
                alarmResetCounter = 0;
            }
        }
        return msg;
    }
    
    // Message sending for hop msg is complete
    event result_t SendHopMsg.sendDone(TOS_MsgPtr msg, result_t success) {
        hopMsgLocked = FALSE;
        return SUCCESS;
    }
    
    // Message sending for alarm msg is complete
    event result_t SendAlarmMsg.sendDone(TOS_MsgPtr msg, result_t success) {
        alarmMsgLocked = FALSE;
        return SUCCESS;
    }
    
    // Read the ADC data
    async event result_t ADC.dataReady(uint16_t data) {
        atomic tsr = data;
        return SUCCESS;
    }
}
