/**
 * Wireless Networks  Sensor Project
 * 03/30/07
 * Members: Chirayu Oza, Gokulakrishnan Ramakrishnan, Vishal Kamat
 

includes MyMsg;

module MultiHopM { 
  provides 	{
    interface StdControl;
  }
  uses {
    interface Leds;
    interface Timer;
    interface SendMsg as SendMyMsg;
    interface ReceiveMsg as ReceiveMyMsg;
    interface StdControl as CommControl;
    interface Random;
  }
}

/* Module Implementation */

implementation {
  // module scoped variables
  TOS_MsgPtr msgSend;       //Pointer to a message to send
  TOS_MsgPtr msgRecv;       //Pointer to a received message
  TOS_Msg buf;              //Buffer for a message to send
  uint16_t parent;          //This mote's parent
  uint8_t mote_count;       //Number of motes below this mote
  uint8_t hops_from_base;   //Number of hops from the basestation
  uint8_t status;           //System status
  uint8_t retrans_count;    //Number of retransmission attempts
  uint8_t child_responses;  //Number of children that sent this mote a COUNT
  uint8_t children;         //Number of children for this mote
  uint8_t motes[MAX_MOTES]; //Entry is mote_count for a given child mote.

 /**
  * Method used to send a message. A unique message is printed for each type.
*/
  
void sendMsg(uint16_t source, uint16_t dest, uint8_t seqnum, uint8_t type, uint8_t count){
    struct MyMsg *bbuf = (struct MyMsg *) msgSend->data;
    bbuf->source = source;
    bbuf->dest = dest;
    bbuf->seqno = seqnum;
    bbuf->type = type;
    bbuf->hop_count = count;
    switch(type){
      case DISCOVERY:
        dbg(DBG_USR1, "Sent DISCOVERY: source %d to %d, sqn %d, type %d, hc %d\n", bbuf->source, bbuf->dest, bbuf->seqno, bbuf->type, bbuf->hop_count);
        break;
      case REPLY:
        dbg(DBG_USR1, "Sent REPLY: source %d to %d, sqn %d, type %d, hc %d\n", bbuf->source, bbuf->dest, bbuf->seqno, bbuf->type, bbuf->hop_count);
        break;
      case REPLY_ACK:
        dbg(DBG_USR1, "Sent REPLY_ACK: source %d to %d, sqn %d, type %d, hc %d\n", bbuf->source, bbuf->dest, bbuf->seqno, bbuf->type, bbuf->hop_count);
        break;
      case COUNT:
        dbg(DBG_USR1, "Sent COUNT: source %d to %d, sqn %d, type %d, hc %d\n", bbuf->source, bbuf->dest, bbuf->seqno, bbuf->type, bbuf->hop_count);
        break;
      case COUNT_ACK:
        dbg(DBG_USR1, "Sent COUNT_ACK: source %d to %d, sqn %d, type %d, hc %d\n", bbuf->source, bbuf->dest, bbuf->seqno, bbuf->type, bbuf->hop_count);
        break;
    }
    call SendMyMsg.send(bbuf->dest, sizeof(struct MyMsg), &buf);
    call Leds.redToggle();
  }

 /** Task that processes received messages. **/

  task void processMsg(){
    struct MyMsg * mmsg = (struct MyMsg *) msgRecv->data;
    switch (mmsg->type){
      case DISCOVERY:

        if(TOS_LOCAL_ADDRESS != BASESTATION){
          //Reset values for each Mote Count attempt by the basestation.
          if(status == STANDBY){
            int i;
            parent = TOS_LOCAL_ADDRESS;
            for(i=0; i < MAX_MOTES; i++)
              motes[i] = 0xff;
            mote_count = 0;
            hops_from_base = 0;
            retrans_count = 0;
            children = 0;
            child_responses = 0;
          }
          if(parent == TOS_LOCAL_ADDRESS){
            parent = mmsg->source;
            hops_from_base = mmsg->hop_count + 1;
          }
          if(mmsg->source == parent){
            dbg(DBG_USR1, "Recv DISCOVERY: source %d, sqn %d, type %d, hc %d\n", mmsg->source, mmsg->seqno, mmsg->type, mmsg->hop_count);
            sendMsg(TOS_LOCAL_ADDRESS, TOS_BCAST_ADDR, mmsg->seqno, mmsg->type, hops_from_base);
            status = REPLY_MODE;
            call Timer.start(TIMER_ONE_SHOT, ((call Random.rand()%1031)+REPLY_TIME)/hops_from_base);
          }
        }
        else{ //Some nodes heard the DISCOVERY so BS should just wait for REPLYs.
          status = STANDBY;
          call Timer.stop();
        }
        break;
      case REPLY:

        if(mmsg->hop_count > hops_from_base){
          dbg(DBG_USR1, "Recv REPLY: source %d, sqn %d, type %d, hc %d\n", mmsg->source, mmsg->seqno, mmsg->type, mmsg->hop_count);
          if(motes[mmsg->source] == 0xff){
            children++;
            motes[mmsg->source] = 0;
            dbg(DBG_USR1, "Updated Child count %d\n", children);
          }
          sendMsg(TOS_LOCAL_ADDRESS, mmsg->source, mmsg->seqno, REPLY_ACK, hops_from_base);
        }
        break;
      case REPLY_ACK:

        if(TOS_LOCAL_ADDRESS != BASESTATION){
          if(mmsg->source == parent){
            if(TOS_LOCAL_ADDRESS == mmsg->dest){
              dbg(DBG_USR1, "Recv REPLY_ACK: source %d, sqn %d, type %d, hc %d\n", mmsg->source, mmsg->seqno, mmsg->type, mmsg->hop_count);
              dbg(DBG_USR1, "Child Count %d\n", children);
              if(children == 0){
                call Timer.start(TIMER_ONE_SHOT, (call Random.rand()%1031)+NO_CHILD_COUNT_TIME);
                status = COUNT_MODE;
              }
              else{
                call Timer.stop();
                status = STANDBY;
              }
              retrans_count = 0;
            }
          }
        }
        break;
      case COUNT:

        if(TOS_LOCAL_ADDRESS == mmsg->dest){
          dbg(DBG_USR1, "Recv COUNT: source %d, sqn %d, type %d, hc %d\n", mmsg->source, mmsg->seqno, mmsg->type, mmsg->hop_count);
          if(motes[mmsg->source] == 0){
            child_responses++;
          }
          if(motes[mmsg->source] < mmsg->hop_count){
            mote_count = mote_count - motes[mmsg->source] + mmsg->hop_count;
            motes[mmsg->source] = mmsg->hop_count;
            dbg(DBG_USR1, "Update: child_responses %d, children %d, motes %d\n", child_responses, children, mote_count);
          }
          sendMsg(TOS_LOCAL_ADDRESS, mmsg->source, mmsg->seqno, COUNT_ACK, children+1);
        }
        if(child_responses == children){
          if(TOS_LOCAL_ADDRESS == BASESTATION){
            call Timer.stop();
            call Timer.start(TIMER_ONE_SHOT, BS_WAIT_TO_FINISH);
            status = TIME_TO_FINISH;
          }
          else{
            status = COUNT_MODE;
            call Timer.stop();
            call Timer.start(TIMER_ONE_SHOT, ((call Random.rand()%1031)+ALL_CHILD_COUNT_TIME)/hops_from_base);
          }
        }
        else{
          if(TOS_LOCAL_ADDRESS != BASESTATION){
            status = COUNT_MODE;
            call Timer.stop();
            call Timer.start(TIMER_ONE_SHOT, ((call Random.rand()%1031)+SOME_CHILD_COUNT_TIME)/hops_from_base);
          }
        }
      case COUNT_ACK:

        if(TOS_LOCAL_ADDRESS != BASESTATION){
          if(mmsg->source == parent){
            if(TOS_LOCAL_ADDRESS == mmsg->dest){
              dbg(DBG_USR1, "Recv COUNT_ACK: source %d, sqn %d, type %d, hc %d\n", mmsg->source, mmsg->seqno, mmsg->type, mmsg->hop_count);
              call Timer.stop();
              status = STANDBY;
            }
          }
        }
        break;
    }
  }


  event TOS_MsgPtr ReceiveMyMsg.receive(TOS_MsgPtr pmsg){
    msgRecv = pmsg;
    call Leds.yellowToggle();
    post processMsg();
    return pmsg;
  }


  event result_t SendMyMsg.sendDone(TOS_MsgPtr mmsg, result_t success){
    return SUCCESS;
  }

  /** When a timer is fired, a mote will react based on its current state **/
  
  event result_t Timer.fired() {
    int i;
    dbg(DBG_USR1, "Timer! Doing Something!\n");
    call Leds.greenToggle();
    switch(status){
      case TIME_TO_FINISH: /** Base Station will print out the mote count **/
        dbg(DBG_USR1, "Mote Count = %d\n", mote_count);
        status = TIME_TO_START;
        call Timer.start(TIMER_ONE_SHOT, BS_RESTART_TIME);
        break;
      case TIME_TO_START: /** Base Station will start the mote count process **/
        for(i=0; i < MAX_MOTES; i++)
          motes[i] = 0xff;
        parent = TOS_LOCAL_ADDRESS;
        mote_count = 0;
        hops_from_base = 0;
        retrans_count = 0;
        children = 0;
        child_responses = 0;
        sendMsg(TOS_LOCAL_ADDRESS, TOS_BCAST_ADDR, 0, DISCOVERY, hops_from_base);
        //call Timer.start(TIMER_REPEAT, DISCOVER_RETRY_TIME);
        break;
      case REPLY_MODE: 

        if(retrans_count <= MAX_RETRANS){
          retrans_count++;
          sendMsg(TOS_LOCAL_ADDRESS, parent, retrans_count, REPLY, hops_from_base);
          call Timer.start(TIMER_ONE_SHOT, (call Random.rand()%1031)+REPLY_RETRANS_TIME);
        }
        else{
          call Timer.stop();
          status = STANDBY;
        }
        break;
      case COUNT_MODE:

        if(retrans_count <= MAX_RETRANS){
          retrans_count++;
          sendMsg(TOS_LOCAL_ADDRESS, parent, retrans_count, COUNT, mote_count+1);
          call Timer.start(TIMER_ONE_SHOT, (call Random.rand()%1031)+COUNT_RETRANS_TIME);
        }
        else{
          call Timer.stop();
          status = STANDBY;
        }
        break;
    }  
    return SUCCESS;
  }

  command result_t StdControl.init() {
    msgSend = &buf;
    if(TOS_LOCAL_ADDRESS == BASESTATION)
      status = TIME_TO_START;
    else
      status = STANDBY;
    dbg(DBG_USR1, "INIT!\n");
    call Random.init();
    return rcombine(call CommControl.init(), call Leds.init());
  }
 
/** start communication layer and basestation timer **/
  command result_t StdControl.start(){
    result_t retval; 
    dbg(DBG_USR1, "START\n");
    if(TOS_LOCAL_ADDRESS == BASESTATION)
      retval = call Timer.start(TIMER_ONE_SHOT, BS_START_TIME);
    else retval = SUCCESS;
    return rcombine (call CommControl.start(), retval);
  }

 /** stop communication layer and any remaining timers **/
  command result_t StdControl.stop(){
    return rcombine(call CommControl.stop(), call Timer.stop());
  } 
}
