;;;; This program is a major revision of a program whose original header
;;;; appears below.  Revisions completed July 30, 1992, by Ronald J. Williams.
;;;;
;;;; The main modifications made are as follows:
;;;;
;;;; 1. Uniform interface provided for test data so that this program can
;;;;    be used as one of a suite of classification-learning programs
;;;; 2. User routines standardized to be similar to other programs in this
;;;;    suite.
;;;; 3. Altered to handle correctly case when negative examples presented
;;;;    initially,  Approach used is to not maintain an explicit list of
;;;;    all specializations until either a positive example is presented or
;;;;    it is determined that there are only a small number (<= 15) left.
;;;;    Makes use of the fact that Until a positive example has appeared
;;;;    the set of specializations is just the complement of the set of all
;;;;    negative examples.  (Observation added 3/27/2003: This is not
;;;;    not really necessary; a simpler approach is just to augment pure
;;;;    conjunctive concept descriptions with the empty set, as is done
;;;;    in Mitchell's book.)
;;;;
;;;; Notes:
;;;; 1. I'm not sure if this program works properly if training examples have
;;;;    don't-cares in them. (But this functionality is not critical for now.)
;;;; 2. Would be nice to modify this so it can handle hierarchical
;;;;    attribute values.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;; Original Header ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; Copyright 1992 Patrick H. Winston.  All rights reserved.
;;;; 3.16 Beta, copied from master file on 8 Jun 92       
;;;; 
;;;; This software is licensed by Patrick H. Winston (licensor) for
;;;; instructional use with the textbooks ``Artificial Intelligence,'' by
;;;; Patrick H. Winston, and ``Lisp,'' by Patrick H. Winston and Berthold
;;;; K. P. Horn.  Your are free to make copies of this software and
;;;; modify it for such instructional use as long as:
;;;; 1. You keep this notice intact.
;;;; 2. You cause any modified files to carry a prominent notice stating
;;;;    that you modified the files and the date of your modifications.
;;;; This software is licensed ``AS IS'' without warranty and the licensor
;;;; shall have no liability for any alleged defect or damages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#|Example of how to use this program.
(load "...-data")			;should call define-training-data
(init-version-space ...)		;can select concept to use
(train-version-space ...)		;for batch training (optional)
(test-version-space ...)		;to classify an example

(assimilate-pair ...)			;these may also be used
(assimilate-positive-example ...)	; optionally, to train
(assimilate-negative-example ...)	;  on individual examples
|#

;;;; DYNAMIC VARIABLES

(defvar *generalizations*)
(defvar *specializations*)

(defvar *version-space-data*)
(defvar *concept*)
(defvar *not-concept*)
(defvar *attribute-number*)
(defvar *nbr-specializations*)		;only kept up to date when implicit;
					; also, value is only correct if no
					;  repeated negative training examples
;;;; USER-LEVEL PROCEDURES

(defun init-version-space (&optional concept)
  (convert-data-to-version-space-form concept)
  (setf *generalizations* (list (mapcar #'(lambda (x) '?) *input-ranges*)))
  (setf *specializations* 'implicit)
  (setf *nbr-specializations*
	(reduce #'* (mapcar #'length *input-ranges*)))
  (format t "~%Starting with 1 generalization and ~a specialization~:p."
	  *nbr-specializations*)
  'done
  )

(defun train-version-space ()		;run on whole set of training data
  (if (not (eql (test-versions) 'inconsistent))
      (dolist (pair *version-space-data* t)
	 (assimilate-pair pair)
	 (if (eql (test-versions) 'inconsistent) (return nil)))))

(defun test-version-space (input-att-vec)
  (when (check-att-vector input-att-vec)
    (if (eql (test-versions) 'inconsistent)
	(format t "~%Cannot classify any examples--version space is empty.")
	(case (test-example input-att-vec)
	      ((t) (format t "~%Answer: ~a" *concept*))
	      ((nil) (format t "~%Answer: ~a" *not-concept*))
	      ((?) (format t "~%Answer: Could be either ~a or ~a"
			   *concept* *not-concept*))
	      ))
    (values)))

(defun assimilate-pair (io-pair)	;run on one input-output pair
  (if (eql (nth *attribute-number* (second io-pair)) *concept*)
      (assimilate-positive-example (first io-pair))
      (assimilate-negative-example (first io-pair))
      ))

(defun assimilate-positive-example (example) ;example is input attribute vector
  "
  Purpose:	Learn from a positive example.
  Remarks:	Much like ASSIMILATE-NEGATIVE-EXAMPLE.
  "
  (if (listp *specializations*)
      ;;If there are explicit specializations already ...
      (progn
	;;Use the example to generalize the specific models:
	(setf *specializations*
	      (mapcan #'(lambda (x) (generalize x example)) *specializations*))
	;;Remove any specific model that is not a specialization
	;;of a general model:
	(setf *specializations*
          (remove-if-not
	    #'(lambda (x) (more-specific-than-one-of-p x *generalizations*))
		*specializations*))
	;;Remove any specific model that is more general
	;;than another specific model:
	(setf *specializations*
	      (get-rid-of-generalizations *specializations*)))
    ;;If there are no explicit specializations yet ...
    (setf *specializations* (list example)))
  ;;Remove any generalization that does not match the example:
  (setf *generalizations*
	(remove-if-not #'(lambda (x) (match x example)) *generalizations*))
  (print-result example "positive"))

(defun assimilate-negative-example (example) ;example is input attribute vector
  "
  Purpose:	Learn from a negative example.
  Remarks:	See ASSIMILATE-POSITIVE example for comments. 
  "
  (setf *generalizations*
	(mapcan #'(lambda (x) (specialize x example))
		*generalizations*))
  (if (listp *specializations*)		;if have explicit specializations
      (setf *generalizations*
	    (remove-if-not
	     #'(lambda (x) (more-general-than-one-of-p x *specializations*))
	     *generalizations*)))
  (setf *generalizations* (get-rid-of-specializations *generalizations*))
  (if (listp *specializations*)		;if have explicit specializations
      (setf *specializations*
	    (remove-if #'(lambda (x) (match x example)) *specializations*))
      (if (<= (decf *nbr-specializations*) 15) ;else if not too many
	  (setf *specializations*	;make them explicit
		(generate-all-instances
		  *input-ranges*
		  *generalizations*))))
  (print-result example "negative"))
  
;;;; AUXILIARY PROCEDURES

(defun convert-data-to-version-space-form (concept)
  ;; If concept is specified, find the first 2-valued output attribute in
  ;; which it appears as a value, or, if no such 2-valued attribute, the
  ;; first attribute in which it appears as a value.  If concept not
  ;; specified, find first two-valued attribute and take the second value
  ;; as the concept.  Otherwise, take the first value of the first attribute
  ;; as the concept.
  (setq *version-space-data* nil)
  (let (range
	not-concept
	attribute-number
	(binary-attributes
	 (remove-if-not #'(lambda (spec) (= (length spec) 2)) *output-ranges*))
	)
    (cond (concept
	   (setq range (find-if #'(lambda (spec) (member concept spec))
				binary-attributes))
	   (if (null range)
	       (setq range (find-if #'(lambda (spec) (member concept spec))
				    *output-ranges*)))
	   (unless range
		   (format t "~%No output attribute can have value ~a"
			   concept)
		   (setq *concept* nil)))
	  (binary-attributes (setq range (first binary-attributes)))
	  (t (setq range (first *output-ranges*)))
	  )
    (when range
	  (setq attribute-number (position range *output-ranges*))
	  (cond ((= (length range) 2)
		 (if concept 
		     (setq not-concept (car (remove concept range)))
		     (progn (setq concept (second range))
			    (setq not-concept (first range)))))
		(t
		 (unless concept
			 (setq concept (first range)))
		 (setq not-concept
		       (make-symbol
			(concatenate 'string "NOT-"
				     (princ-to-string concept))))))
	  (setq *version-space-data* *training-data*)
	  (setq *concept* concept)
	  (setq *not-concept* not-concept)
	  (setq *attribute-number* attribute-number)
	  (format t "~%Preparing to learn concept ~a" concept)
	  (if (> (length *output-ranges*) 1)
	      (format t " for output attribute number ~a" attribute-number))
	  (values)
	  )))

(defun test-example (example)
  "
  Purpose:	Test example.
  "
  (format t "~%")
  ;;Self explanatory:
  (cond ((and (listp *specializations*)
	      (every #'(lambda (x) (match x example)) *specializations*))
	 (format t "The example ~a~
		 ~%matches all specializations; ~
		 it must be a positive example." example)
	 t)
	((not (some #'(lambda (x) (match x example)) *generalizations*))
	 (format t "The example ~a~
		 ~%fails to match any generalizations; ~
		 it must be a negative example." example)
	 nil)
	(t (format t "The example ~a~%may be positive or negative." example)
	   '?)))

(defun test-versions ()
  "
  Purpose:	Determine when procedure has converged.
  "
  (cond ((and (listp *specializations*)
	      (= (length *generalizations*) 1)
	      (= (length *specializations*) 1)
	      (equal *generalizations* *specializations*))
	 'converged)
	((or (null *generalizations*) (null *specializations*))
	 'inconsistent)
	))

(defun print-result (example type)
  (format t "~%~%Assimilating ~a example ~a." type example)
  (format t "~%Surviving generalizations:")
  (show-vs-data *generalizations*)
  (format t "~%Surviving specializations:")
  (show-vs-data *specializations*)
  (case (test-versions)
    ((converged) (format t "~%Converged on ~a" (first *generalizations*)))
    ((inconsistent)
     (format t
	     "~%No generalizations or specializations left~
             ---version space is empty.~%~a cannot be pure conjunctive ~
             in the given attributes."
	     *concept*)))
  (values))
		  
(defun show-vs-data (models)
  (if (listp models)
      (dolist (m models) (format t "~%~a" m))
      (format t "~%  All ~a instances consistent with these generalizations."
	      *nbr-specializations*)
      ))

(defun match (model sample)
  "
  Purpose:	Determine if the given sample is an instance of the model.
  "
  (cond ((endp model) t)
	((or (equal '? (first model))
	     (equal (first model) (first sample)))
	 (match (rest model) (rest sample)))
	(t nil)))

(defun generalize (model sample)
  "
  Purpose:	Generalize one model using the sample.
  Remarks:	Generalization only to ? symbol which matches everything;
		there is no class hierarchy.
  "
  (if (match model sample)
      (list model)
    (list (mapcar #'(lambda (x y) (if (equal x y) x '?)) model sample))))

(defun specialize (model sample &aux results)
  "
  Purpose:	Specialize one model using the sample.
  Remarks:	Specialization is to elements in a list of classes;
		there is no class hierarchy.
  "
  (if (match model sample)
      (dotimes (n (length model) results)
	(when (equal '? (nth n model))
	  (dolist (substitution (remove (nth n sample) (nth n *input-ranges*)))
	    (let ((new-model (copy-list model)))
	      (setf (nth n new-model) substitution)
	      (push new-model results)))))
    (list model)))

(defun more-general-than-one-of-p (generalization specializations)
  (some #'(lambda (x) (more-general-than-p generalization x))
	specializations))

(defun more-specific-than-one-of-p (specialization generalizations)
  (some #'(lambda (x) (more-general-than-p x specialization))
	generalizations))

(defun more-general-than-p (generalization specialization)
  "
  Remarks:	Much like MATCH.
  "
  (cond ((endp specialization) t)
	((or (equal '? (first generalization))
	     (equal (first generalization) (first specialization)))
	 (more-general-than-p (rest generalization) (rest specialization)))
	((equal '? (first specialization)) nil)
	(t nil)))

(defun get-rid-of-specializations (models &aux result)
  (dolist (model models result)
    (unless (more-specific-than-one-of-p model (remove model models))
      (push model result))))

(defun get-rid-of-generalizations (models &aux result)
  (dolist (model models result)
    (unless (more-general-than-one-of-p model (remove model models))
      (push model result))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; These procedures are used to create an explicit list of multiple
;;;; specializations.

(defun generate-all-instances (ranges generalizations)
  ;; Generate all instances consistent with the given range specifications
  ;; and list of generalizations.
  (remove-duplicates
    (reduce #'append
	    (mapcar #'(lambda (gen) (gen-instances ranges gen))
		    generalizations))
    :test #'equal))

(defun gen-instances (ranges generalization)
  (cross-product
    (mapcar #'(lambda (range val) (if (eql val '?) range (list val)))
	    ranges
	    generalization)))

(defun cross-product (list-of-lists)
  (if (null list-of-lists)
      (list nil)
      (cons-elts (car list-of-lists)
		 (cross-product (cdr list-of-lists)))
      ))

(defun cons-elts (elts lists)
  (apply #'append
  (mapcar #'(lambda (x)
	      (mapcar #'(lambda (l) (cons x l)) lists))
	  elts)))
