(defvar *input-ranges*)
(defvar *output-ranges*)
(defvar *training-data*)

;;; User routines

(defun define-training-data (input-ranges output-ranges training-data)
  (setq *input-ranges* (mapcar #'remove-duplicates input-ranges))
  (setq *output-ranges* (mapcar #'remove-duplicates output-ranges))
  (setq *training-data*
	(remove-if-not
	  #'(lambda (datum)
	      (let ((in-ok (check-att-vector (first datum) 'input))
		    (out-ok (check-att-vector (second datum) 'output)))
		(and in-ok out-ok)))
	  training-data))
  (format t "~%~a training pattern~:p successfully loaded."
	  (length *training-data*))
  (values))

(defun show-training-data ()
  (dolist (d *training-data* (values))
    (format t "~%~a" d)))

(defun show-ranges ()
  (format t "~%Input:")
  (dolist (r *input-ranges* (format t "~%"))
    (format t "~%~a" r))
  (format t "~%Output:")
  (dolist (r *output-ranges* (format t "~%"))
    (format t "~%~a" r))
  (values))

;;; Auxiliaries

(defun check-att-vector (vec &optional (direction 'input))
  (let ((spec-list
	  (if (eql direction 'input) *input-ranges* *output-ranges*))
	(att-number -1)
	(ok-flag t)
	)
    (mapc #'(lambda (value spec)	;will not detect length mismatch
	      (incf att-number)
	      (unless (check-value-against-spec value spec)
		(when ok-flag
		  (format t "~%Bad ~a: ~a"
			  (if (eql direction 'input) "input" "output")
			  vec))
		(format t "~%Attribute ~a value ~a does not match spec ~a"
			att-number
			value
			spec)
		(setq ok-flag nil)))
	  vec
	  spec-list)
    ok-flag
    ))

(defun check-value-against-spec (value spec)
  (if (num-range-spec-p spec)
      (and (numberp value)
	   (<= (reduce #'min spec) value (reduce #'max spec)))
      (member value spec)
      ))

(defun num-range-spec-p (value-list)
  (and (= (length value-list) 2)
       (numberp (first value-list))
       (numberp (second value-list))
       (/= (first value-list) (second value-list))
       ))

(defun scale-to-0-1 (x num-range-spec)
  (let* ((lower (reduce #'min num-range-spec))
	 (upper (reduce #'max num-range-spec))
	 (scaled-x (/ (- x lower) (- upper lower))))
    (if (typep scaled-x 'ratio)
	(float scaled-x)
        scaled-x)
    ))

(defun unscale-from-0-1 (x num-range-spec)
  (let ((lower (reduce #'min num-range-spec))
	(upper (reduce #'max num-range-spec)))
    (+ (* x (- upper lower)) lower)
    ))


;;; These routines all return an association list of item/number pairs
;;; (item number), with exactly one such list for each unique item.
;;; Tally-scores and tally-and-sort-scores take as input a corresponding
;;; association list.

(defun tally-items (items &key (test #'eql))
  (tally-scores (mapcar #'(lambda (x) (list x 1)) items) :test test))

(defun tally-and-sort-items (items &key (test #'eql))
  (tally-and-sort-scores (mapcar #'(lambda (x) (list x 1)) items) :test test))

(defun tally-and-sort-scores (item-score-pairs &key (test #'eql))
  (sort (tally-scores item-score-pairs :test test) #'> :key #'second))

(defun tally-scores (item-score-pairs &key (test #'eql))
  (remove-duplicates	    ;tricky: has side effect of accumulating scores
   item-score-pairs
   :test #'(lambda (p1 p2)
	     (if (funcall test (first p1) (first p2))
		 (incf (second p2) (second p1))))))

;;; This converts the kind of output produced by several of the programs
;;; into a slightly more readable form.

(defun prettify-output (list-of-alists)
  (let ((att-num -1))
    (dolist (alist (round-all-floats list-of-alists) (values))
      (if (> (length *output-ranges*) 1)
	  (format t "~%Output attribute ~a:" (incf att-num)))
      (format t "~%~a" alist))))

(defun round-all-floats (s)		;floats are converted to ~4,2f format
  (cond ((null s) nil)
	((listp s)
	 (cons (round-all-floats (car s))
	       (round-all-floats (cdr s))))
	((typep s 'float) (format nil "~4,2f" s))
	(t s)))

;;; Should also put accessor functions for input, output patterns here.
